/*
 * cNetworkLayer.cpp
 *
 * THE network layer.
 */
#include "cNetworkLayer.h"
#include "cNetworkLayerParam.h"
#include "Util/gError.h"
#include "Util/cRegisterList.h"
#include "Protocol Stack/cProtocolStack.h"
#include "Endpoint/cEndpointFactory.h"

#include <memory.h>
#include <malloc.h>
#include <ws2tcpip.h>
#include <stdio.h>

const unsigned int cNetworkLayerParam::UNRELIABLE_POINT_TO_POINT = 0x01;
const unsigned int cNetworkLayerParam::UNRELIABLE_BROADCAST		 = 0x02;
const unsigned int cNetworkLayerParam::UNRELIABLE_MULTICAST		 = 0x04;
const unsigned int cNetworkLayerParam::RELIABLE_POINT_TO_POINT	 = 0x08;

/*
 * cNetworkLayer::Deliver()
 *
 * Purpose:	Delivery interface function.
 * IN:		sender		-> The sender of the msg
 *			buffer		-> The buffer delivered up.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::Deliver(cEndpoint* sender, cMsgBuffer* buffer, int messageType)
{
	cIterator*	iter;
	cDeliver*	deliver;
	int			size;

	// Remove and process header
	char* header;
	buffer->GetPayload((void **)&header, &size);

	sender = cEndpointFactory::AllocEndpoint(ENDPOINT_TYPE_IP);
	if(!sender)
	{
		gError("Unable to allocate endpoint on receive.", __LINE__, __FILE__);
	}
	sender->Deserialize(header, &size);
	buffer->RemoveHeader(sender->GetSize());

	sender->AddRef();

	// Deliver to callbacks.
	iter = mRegisterList->GetIterator();
	while(!iter->Done())
	{
		deliver = (cDeliver *)iter->GetData();
		deliver->Deliver(sender, buffer, messageType);
		iter->GetNext();
	}

	sender->Release();

	return true;
}

/*
 * cNetworkLayer::Init()
 *
 * Purpose:	Initializes the network layer.
 * IN:		layerBelow	-> The layer below this layer.
 *			param		-> The parameters intended for this layer.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::Init(cLayer* layerBelow, cParam* param)
{
	mRegisterList = NULL;
	cNetworkLayerParam*	nParam = (cNetworkLayerParam *)param;

	// Init our pointer to the protocol stack.
	mProtocolStack = param->mProtocolStack;

	// Set up register list.
	mRegisterList = new cRegisterList(8);	// CHANGE!!!
	if(!mRegisterList)
	{
		return false;
	}

	// Set up the error callback list.
	mErrorCallbackList = new cRegisterList(8); // CHANGE!!!
	if(!mErrorCallbackList)
	{
		return false;
	}

	// Copy layer info from below
	mLayerBelow = layerBelow;
	if(layerBelow)
	{
		cLayerInfo* belowInfo;
		belowInfo = mLayerBelow->GetLayerInfo();
		mLayerInfo = *belowInfo;
	}
	mLayerInfo.SetLayerBelow(mLayerBelow);
	mLayerInfo.AddHeaderSize(mLocalEndpoint.GetSize());

	// Set up the network
	if(!_SetupNetwork(nParam))
	{
		gError("Unable to set up the network.", __LINE__, __FILE__);
		return false;
	}

	int epBufSize = sizeof(mMyEpSer);
	if(!mLocalEndpoint.Serialize(mMyEpSer, &epBufSize))
	{
		gError("Unable to fit local endpoint into serialized storage buffer.", __LINE__, __FILE__);
		return false;
	}

	// Register with layer below
	if(mLayerBelow)
	{
		if(!layerBelow->RegisterDeliverCallback(&mLayerBelowHandle, this))
		{
			gError("Unable to register with layer below.", __LINE__, __FILE__);
			return false;
		}
	}
	return true;
}

/*
 * cNetworkLayer::Cleanup()
 *
 * Purpose:	Cleans up the network layer.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is no longer useable.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::Cleanup()
{
	if(mLayerBelow)
	{
		mLayerBelow->UnregisterDeliverCallback(mLayerBelowHandle);
	}

	// Delete the register list.
	if(mRegisterList)		{ delete mRegisterList; mRegisterList = NULL; }

	// Delete error callback list.
	if(mErrorCallbackList)	{ delete mErrorCallbackList; mErrorCallbackList = NULL; }

	_ShutdownNetwork();

	return true;
}

/*
 * cNetworkLayer::Send()
 *
 * Purpose:	Prints out the msg to send and passes it to layer below.
 * IN:		dest		-> The message destination.
 *			buffer		-> the actual message.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::Send(cGroup* dest, cMsgBuffer* buffer, int messageType)	
{
	char*	outBuf;
	int		size;
	int		i;
	cEndpoint* ep;
	cIterator* iter;
	cIPEndpoint	tempEp;
	cIPEndpoint* iep;

	// Add the header
	if(!buffer->AddHeader(&mMyEpSer[0], mLocalEndpoint.GetSize()))
	{
		return false;
	}

	// Send message on the network
	buffer->GetPayload((void **)&outBuf, &size);

	iter = dest->GetIterator();
	while(!iter->Done())
	{
		ep = (cEndpoint *)iter->GetData();
		if(ep->GetType() == ENDPOINT_TYPE_IP)
		{
			iep = (cIPEndpoint *)ep;
			switch(messageType)
			{
			  case MSG_TYPE_BCAST:
//				memcpy(&tempEp.mAddress, &(iep->mAddress), sizeof(tempEp.mAddress));
//				tempEp.mAddress.sin_addr.s_addr |= ~iep->mSubnetMask;
//				ep = &tempEp;
				if(!_SendDatagram(&mMulticastSocket[0], (cIPEndpoint *)ep, outBuf, size))
				{
					gError("Unable to send broadcast datagram.", __LINE__, __FILE__);
					buffer->RemoveHeader(mLocalEndpoint.GetSize());
					_DeliverErrorCallback(buffer, ASYNCH_ERROR_SND_FAIL);
					return false;
				}
				break;

			  case MSG_TYPE_MULTICAST:
				for(i = 0; i < MAX_MCAST_INTERFACES; i++)
				{
					if(mMulticastSocket[i] == INVALID_SOCKET)
					{
						break;	// Stop sending when run out of multicast interfaces
					}
					if(!_SendDatagram(&mMulticastSocket[i], (cIPEndpoint *)ep, outBuf, size))
					{
						gError("Unable to send multicast datagram.", __LINE__, __FILE__);
						buffer->RemoveHeader(mLocalEndpoint.GetSize());
						_DeliverErrorCallback(buffer, ASYNCH_ERROR_SND_FAIL);
						return false;
					}
				}
				break;

			  case MSG_TYPE_UNICAST:
				if(!_SendDatagram(&mBroadcastSocket, (cIPEndpoint *)ep, outBuf, size))
				{
					gError("Unable to send unicast datagram.", __LINE__, __FILE__);
					buffer->RemoveHeader(mLocalEndpoint.GetSize());
					_DeliverErrorCallback(buffer, ASYNCH_ERROR_SND_FAIL);
					return false;
				}
				break;
			}
		}
		iter->GetNext();
	}

	// Send to layer below.
	if(mLayerBelow)
	{
		mLayerBelow->Send(dest, buffer, messageType);
	}
	buffer->RemoveHeader(mLocalEndpoint.GetSize());

	return true; 
}

/*
 * cNetworkLayer::RegisterDeliverCallback()
 *
 * Purpose:	Registers the deliver callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool cNetworkLayer::RegisterDeliverCallback(cHandle* handle, cDeliver* callback) 
{ 
	handle->mLayer = this;
	return mRegisterList->AddObject(handle, (cObject *)callback);
}

/*
 * cNetworkLayer::UnregisterDeliverCallback()
 *
 * Purpose:	Unregisters the deliver callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::UnregisterDeliverCallback(cHandle handle)	
{ 
	if(handle.mLayer == this)
	{
		return mRegisterList->RemoveObject(handle);
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterDeliverCallback(handle);
	}
	else
	{
		return false;
	}
}

/*
 * cNetworkLayer::RegisterErrorCallback()
 *
 * Purpose:	Registers the asynchronous error callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool cNetworkLayer::RegisterErrorCallback(cHandle* handle, cErrorCallback* callback)
{ 
	handle->mLayer = this;
	return mErrorCallbackList->AddObject(handle, (cObject *)callback);
}

/*
 * cNetworkLayer::UnregisterErrorCallback()
 *
 * Purpose:	Unregisters the error callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::UnregisterErrorCallback(cHandle handle)
{ 
	if(handle.mLayer == this)
	{
		return mErrorCallbackList->RemoveObject(handle);
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterErrorCallback(handle);
	}
	else
	{
		return false;
	}
}


/*
 * cNetworkLayer::_ShutdownNetwork()
 *
 * Purpose:	Shuts down the network and frees all network resources.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The networking stuff should be gracefully shut down.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::_ShutdownNetwork()
{

	if(mBroadcastSocket != INVALID_SOCKET)
	{
		mProtocolStack->RemoveEvent(mBroadcastEvent);
		closesocket(mBroadcastSocket);
		mBroadcastSocket = INVALID_SOCKET;		
		WSACloseEvent(mBroadcastEvent);
	}

	for(int i = 0; i < MAX_MCAST_INTERFACES; i++)
	{
		if(mMulticastSocket[i] != INVALID_SOCKET)
		{
			mProtocolStack->RemoveEvent(mMulticastEvent[i]);
			closesocket(mMulticastSocket[i]);
			mMulticastSocket[i] = INVALID_SOCKET;
			WSACloseEvent(mMulticastEvent[i]);
		}
	}

	//mProtocolStack->RemoveLocalAddress(&mLocalAddress);

	if(WSACleanup() != 0)
	{
		gError("Unable to clean up winsock.", __LINE__, __FILE__);
	}
	return true;
}

/*
 * cNetworkLayer::_SetupMulticast()
 *
 * Purpose:	Sets up the local multicast sockets.
 * IN:		-
 * OUT:		-
 * Cond:	The mBroadcastSocket should be set up.
 * PostCnd:	Multicast should be set up.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::_SetupMulticast(cNetworkLayerParam* param)
{
	int  nRet;
	int  bSetOption = 1;
	bool protoFound = false;
	int	 goodProtocol;
	WSAPROTOCOL_INFO	*protoBuf = NULL;
	INTERFACE_INFO   pInterfaceInfo[MAX_MCAST_INTERFACES], *pInfo;
	DWORD	 bytesReturned;
	int		 numInterfacesFound;
	DWORD	 bufLen = 0;
	SOCKET	 tempSocket = INVALID_SOCKET;

	// Find the right protocol to use.
	nRet = WSAEnumProtocols(NULL, protoBuf, &bufLen);
	protoBuf = (WSAPROTOCOL_INFO *)malloc(bufLen);
	if(!protoBuf)
	{
		return false;
	}
	nRet = WSAEnumProtocols(NULL, protoBuf, &bufLen);
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to enumerate multicast protocols.", __LINE__, __FILE__);
		return false;
	}
	for(int i = 0; i < nRet; i++)
	{
		if( (protoBuf[i].iProtocol == IPPROTO_UDP) &&
			(XP1_SUPPORT_MULTIPOINT == (protoBuf[i].dwServiceFlags1 & XP1_SUPPORT_MULTIPOINT)) &&
		   !(XP1_MULTIPOINT_CONTROL_PLANE == (protoBuf[i].dwServiceFlags1 & XP1_MULTIPOINT_CONTROL_PLANE)) &&
		   !(XP1_MULTIPOINT_DATA_PLANE == (protoBuf[i].dwServiceFlags1 & XP1_MULTIPOINT_DATA_PLANE))
		   )
		{
			protoFound = true;
			goodProtocol = i;
			break;
		}
	}
	if(!protoFound)
	{
		gError("Unable to find supporting multicast protocol.", __LINE__, __FILE__);
		return false;
	}

	// Enumerate all of the interfaces.
	tempSocket = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if(tempSocket == INVALID_SOCKET)
	{
		gError("Unable to create temporary socket", __LINE__, __FILE__);
		return false;
	}
	nRet = WSAIoctl(tempSocket, 
					SIO_GET_INTERFACE_LIST, // command
					NULL,					// Input buffer
					0,						// Size of input buffer
					pInterfaceInfo,			// Pointer to output buffer
					sizeof(pInterfaceInfo),	// size of output buffer
					&bytesReturned,			// actual number of bytes output
					NULL,					// pointer to overlapped structure -- ignore
					NULL);					// pointer to overlapped completion routine -- ignore
	if (nRet == SOCKET_ERROR)
	{
		gError("Unable to enumerate the interface list.", __LINE__, __FILE__);
		return false;
	}
	closesocket(tempSocket);
	numInterfacesFound = bytesReturned / sizeof(INTERFACE_INFO);
	for(i = 0; i < numInterfacesFound; i++)
	{
		pInfo = &pInterfaceInfo[i];
		if(!(pInfo->iiFlags & IFF_UP) || !(pInfo->iiFlags & IFF_MULTICAST) || (pInfo->iiFlags & IFF_LOOPBACK))
		{
			continue;	// Ignore: interface that is down || one that can't mcast || loopback of 127.0.0.1
		}

		// Set port
		pInfo->iiAddress.AddressIn.sin_port = htons(mMulticastPort);

		// Create mcast socket.
		mMulticastSocket[i] = WSASocket( AF_INET, 
										 SOCK_DGRAM, 
										 IPPROTO_UDP,
										 &protoBuf[goodProtocol],
										 0,	// No group operation performed
										 WSA_FLAG_MULTIPOINT_C_LEAF | WSA_FLAG_MULTIPOINT_D_LEAF
										);
		if(mMulticastSocket[i] == INVALID_SOCKET)
		{
			gError("Unable to create multicast socket", __LINE__, __FILE__);
			return false;
		}

		// Set mcast socket to be able to re-use the address.
		nRet = setsockopt(mMulticastSocket[i], SOL_SOCKET, SO_REUSEADDR, (const char *)&bSetOption, sizeof(int));
		if(nRet == SOCKET_ERROR) 
		{
			gError("Unable to set socket to be able to re-use existing address.", __LINE__, __FILE__);
			return false;
		}

		// Set mcast socket to broadcast mode.
		nRet = setsockopt(mMulticastSocket[i], SOL_SOCKET, SO_BROADCAST, (const char *)&bSetOption, sizeof(int));
		if(nRet == SOCKET_ERROR) 
		{
			gError("Unable to set socket to broadcast mode.", __LINE__, __FILE__);
			return false;
		}

		// Pump up send and receive buffers!
		int bufsize, optsize;
		optsize = sizeof(int);
		nRet = getsockopt(mMulticastSocket[i], SOL_SOCKET, SO_RCVBUF, (char *)&bufsize, &optsize);
		if(nRet == SOCKET_ERROR)
		{
			gError("Unable to get receive buffer size.", __LINE__, __FILE__);
			return false;
		}
		bufsize = 8*bufsize;
		nRet = setsockopt(mMulticastSocket[i], SOL_SOCKET, SO_RCVBUF, (char *)&bufsize, sizeof(int));
		if(nRet == SOCKET_ERROR)
		{
			gError("Unable to pump up receive buffer to 4x its size.", __LINE__, __FILE__);
			return false;
		}
		nRet = getsockopt(mMulticastSocket[i], SOL_SOCKET, SO_SNDBUF, (char *)&bufsize, &optsize);
		if(nRet == SOCKET_ERROR)
		{
			gError("Unable to get send buffer size.", __LINE__, __FILE__);
			return false;
		}
		bufsize = 8*bufsize;
		nRet = setsockopt(mMulticastSocket[i], SOL_SOCKET, SO_SNDBUF, (char *)&bufsize, sizeof(int));
		if(nRet == SOCKET_ERROR)
		{
			gError("Unable to pump up send buffer to 4x its size.", __LINE__, __FILE__);
			return false;
		}

		// NOTE: The below code fails on NT 4.0 for some reason.  I think JoinLeaf sets the send interface anyway.

		// Set multicast sender interface.
//		nRet = setsockopt(mMulticastSocket[i], IPPROTO_IP, IP_MULTICAST_IF, (char *)&pInfo->iiAddress.AddressIn.sin_addr.s_addr, sizeof(u_long));
//		if(nRet == SOCKET_ERROR)
//		{
//			gError("Unable to set the multicast interface on this socket.", __LINE__, __FILE__);
//			return false;
//		}

		// END NOTE.
		
		// Bind socket
		if ( bind(mMulticastSocket[i], (sockaddr*)&pInfo->iiAddress.Address, sizeof(pInfo->iiAddress.Address)) == SOCKET_ERROR )
		{
			gError("Unable to bind socket.", __LINE__, __FILE__);
			return false;
		}

		// Joinleaf
		SOCKET	temp;
		WSABUF	joinBuf;
		joinBuf.buf = NULL;
		joinBuf.len = 0;
		temp = WSAJoinLeaf(mMulticastSocket[i], (sockaddr*)&(param->mMulticastGroup->mAddress),
						   sizeof(param->mMulticastGroup->mAddress),
						   NULL,
						   &joinBuf, 
						   NULL,	// i don't worry about qos
						   NULL,	// " 
						   JL_BOTH);
		if(temp == INVALID_SOCKET)
		{
			gError("Unable to join multicast group.", __LINE__, __FILE__);
			return false;
		}

		if(!param->mRecvOwnMulticasts)
		{
			// Set mcast so that we don't receive our own through the loopback interface.
			BOOL loopback = FALSE;
			DWORD dwBytes;
			nRet = WSAIoctl(mMulticastSocket[i], SIO_MULTIPOINT_LOOPBACK, &loopback, sizeof(loopback), NULL, NULL, &dwBytes, NULL, NULL);
			if(nRet == SOCKET_ERROR)
			{
				int err = WSAGetLastError();
				gError("Unable to set multicast loopback to false.", __LINE__, __FILE__);
				return false;
			}
		}

		// Get the socket MTS
		unsigned int mts = 0;
		int			 mtsLen = sizeof(mts);
		nRet = getsockopt(mMulticastSocket[i], SOL_SOCKET, SO_MAX_MSG_SIZE, (char *)&mts, &mtsLen);
		if(nRet == SOCKET_ERROR)
		{
			gError("Unable to get mts on socket.", __LINE__, __FILE__);
			return false;
		}
		mLayerInfo.SetMTS(mts - mLocalEndpoint.GetSize());

		// Create event for socket
		mMulticastEvent[i] = WSACreateEvent();
		if(mMulticastEvent[i] == WSA_INVALID_EVENT)
		{
			gError("Unable to create socket event.", __LINE__, __FILE__);
			return false;
		}

		// Set up the event to work.
		nRet = WSAEventSelect(mMulticastSocket[i], mMulticastEvent[i], FD_READ);
		if(nRet == SOCKET_ERROR)
		{
			gError("Unable to event select for broadcast socket.", __LINE__, __FILE__);
			return false;
		}

		// Add event to protocol stack
		if(!param->mProtocolStack->AddEvent(mMulticastEvent[i], this, (void *)&mMulticastSocket[i]))
		{
			gError("Unable to add multicast socket event to protocol stack.", __LINE__, __FILE__);
			return false;
		}
	} /* end for i */

	free(protoBuf);
	return true;
}

/*
 * cNetworkLayer::_SetupBroadcastSocket()
 *
 * Purpose:	Sets up the local socket.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The networking stuff should be set up.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::_SetupBroadcastSocket(cNetworkLayerParam* param)
{
	int nRet;
	int bSetOption = 1;

	// Set port
	mLocalEndpoint.mAddress.sin_port = 0;	// So that it will choose a port for me.

	// Create broadcast socket.
	mBroadcastSocket = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if(mBroadcastSocket == INVALID_SOCKET)
	{
		gError("Unable to create broadcast socket", __LINE__, __FILE__);
		return false;
	}

	// NOTE: mBroadcastSocket is an *old* name.  Should be changed to mIdSocket or something like that.  The mcast socket is used for bcast.

	// Set socket to broadcast mode.
//	nRet = setsockopt(mBroadcastSocket, SOL_SOCKET, SO_BROADCAST, (const char *)&bSetOption, sizeof(int));
//	if(nRet == SOCKET_ERROR) 
//	{
//		gError("Unable to bind socket.", __LINE__, __FILE__);
//		return false;
//	}

	// END NOTE

	// Pump up send and receive buffers!
	int bufsize, optsize;
	optsize = sizeof(int);
	nRet = getsockopt(mBroadcastSocket, SOL_SOCKET, SO_RCVBUF, (char *)&bufsize, &optsize);
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to get receive buffer size.", __LINE__, __FILE__);
		return false;
	}
	bufsize = 8*bufsize;
	nRet = setsockopt(mBroadcastSocket, SOL_SOCKET, SO_RCVBUF, (char *)&bufsize, sizeof(int));
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to pump up receive buffer to 4x its size.", __LINE__, __FILE__);
		return false;
	}
	nRet = getsockopt(mBroadcastSocket, SOL_SOCKET, SO_SNDBUF, (char *)&bufsize, &optsize);
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to get send buffer size.", __LINE__, __FILE__);
		return false;
	}
	bufsize = 8*bufsize;
	nRet = setsockopt(mBroadcastSocket, SOL_SOCKET, SO_SNDBUF, (char *)&bufsize, sizeof(int));
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to pump up send buffer to 4x its size.", __LINE__, __FILE__);
		return false;
	}
		
	// Bind socket
	if ( bind(mBroadcastSocket, (sockaddr*)&mLocalEndpoint.mAddress, sizeof(mLocalEndpoint.mAddress)) == SOCKET_ERROR )
	{
		gError("Unable to bind socket.", __LINE__, __FILE__);
		return false;
	}

	// Get the socket MTS
	unsigned int mts = 0;
	int			 mtsLen = sizeof(mts);
	nRet = getsockopt(mBroadcastSocket, SOL_SOCKET, SO_MAX_MSG_SIZE, (char *)&mts, &mtsLen);
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to get mts on socket.", __LINE__, __FILE__);
		return false;
	}
	mLayerInfo.SetMTS(mts - mLocalEndpoint.GetSize());

	// Create event for socket
	mBroadcastEvent = WSACreateEvent();
	if(mBroadcastEvent == WSA_INVALID_EVENT)
	{
		gError("Unable to create socket event.", __LINE__, __FILE__);
		return false;
	}

	// Set up the event to work.
	nRet = WSAEventSelect(mBroadcastSocket, mBroadcastEvent, FD_READ | FD_CLOSE);
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to event select for broadcast socket.", __LINE__, __FILE__);
		return false;
	}

	// Add event to protocol stack
	if(!param->mProtocolStack->AddEvent(mBroadcastEvent, this, (void *)&mBroadcastSocket))
	{
		gError("Unable to add broadcast socket event to protocol stack.", __LINE__, __FILE__);
		return false;
	}
	return true;
}



/*
 * cNetworkLayer::_SetupNetwork()
 *
 * Purpose:	Sets up the sockets network
 * IN:		param		-> The network parameters given.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The networking stuff should be set up.
 * Return:	true if success, else false.
 */
bool cNetworkLayer::_SetupNetwork(cNetworkLayerParam* param)	
{ 
	int bSetOption = 1;

	// Set sockets to invalid.
	mBroadcastSocket  = INVALID_SOCKET;
	for(int i = 0; i < MAX_MCAST_INTERFACES; i++)
	{
		mMulticastSocket[i] = INVALID_SOCKET;
	}
	mBroadcastPort	  = param->mBroadcastPort;
	mMulticastPort	  = param->mMulticastPort;

	// Start up sockets.
	WSADATA	wsaData;
	if(WSAStartup(0x0202, &wsaData) != 0)
	{
		gError("Unable to startup windows sockets.", __LINE__, __FILE__);
		return false;
	}

	// Get local address
	memset((void *)&mLocalEndpoint.mAddress, 0, sizeof(sockaddr_in));
	mLocalEndpoint.mAddress.sin_family = AF_INET;
	mLocalEndpoint.mAddress.sin_addr.s_addr = _GetHostID();	// Returns in network order :-)
	mLocalEndpoint.mSubnetMask = htonl(param->mSubnetMask);

	// Create broadcast socket if necessary.
	if(param->mNetworkOptions & cNetworkLayerParam::UNRELIABLE_BROADCAST)
	{
		if(!_SetupBroadcastSocket(param))
		{
			gError("Unable to create local non-mcast socket.", __LINE__, __FILE__);
			return false;
		}
	}

	// Create multicast socket if necessary.
	if(param->mNetworkOptions & cNetworkLayerParam::UNRELIABLE_MULTICAST)
	{
		if(!_SetupMulticast(param))
		{
			gError("Unable to set up IP multicast.", __LINE__, __FILE__);
			return false;
		}
	}

	// Fill in our local address.
	int nameSize = sizeof(mLocalEndpoint.mAddress);
	if ( getsockname(mBroadcastSocket, (sockaddr*)&mLocalEndpoint.mAddress, &nameSize) == SOCKET_ERROR )
	{
		gError("Unable to get socket name for local endpoint.", __LINE__, __FILE__);
		return false;
	}

	// Add this address to the local address group
	if(!param->mProtocolStack->AddLocalAddress(&mLocalEndpoint))
	{
		gError("Unable to add local address to protocol stack.", __LINE__, __FILE__);
		return false;
	}

	return true;
}

/*
 * cNetworkLayer::Callback()
 *
 * Purpose: Callback on socket receive.
 * IN:		eventHandle -> The handle for the triggered event.
 *			param		-> The socket to receive on.
 * OUT:		-
 * Return:  -
 */
void cNetworkLayer::Callback(HANDLE eventHandle, void* param)
{
	int nRet;
	unsigned long dataAmt;
	int addressSize;
	sockaddr_in	addr;
	cMsgBuffer*	buf;
	WSANETWORKEVENTS	networkEvents;
	char*		inputBuf;

	// Make sure this is an FD_READ event.
	nRet = WSAEnumNetworkEvents(*((SOCKET *)param), eventHandle, &networkEvents);
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to enumerate what my event was.", __LINE__, __FILE__);
		return;
	}
	if(!(networkEvents.lNetworkEvents & FD_READ))
	{
		gError("Received a non FD_READ event.", __LINE__, __FILE__);
		return;
	}

	// Determine how much data there is to be read in next dgram.
	nRet = ioctlsocket(*((SOCKET *)param), FIONREAD, &dataAmt);
	if(nRet == SOCKET_ERROR)
	{
		gError("Unable to detect how much data is on the socket!", __LINE__, __FILE__);
	}
	if(dataAmt == 0)
	{
		buf = NULL;
		inputBuf = NULL;
		nRet = 0;
	}
	else
	{
		// Allocate a receive buffer.
		buf = mProtocolStack->AllocateBuffer(dataAmt);
		if(!buf)
		{
			gError("Unable to retrieve a buffer to receive into.", __LINE__, __FILE__);
			return;
		}
		buf->GetPayload((void **)&inputBuf, &nRet);
	}
	addressSize = sizeof(addr);
	nRet = recvfrom(*((SOCKET *)param), inputBuf, nRet, 0, (sockaddr *)&addr, &addressSize);
	if(nRet == SOCKET_ERROR)
	{
		if(buf) { buf->Release(); }

		nRet = WSAGetLastError();
		if(nRet == WSAEWOULDBLOCK)
		{ 
			// Should enter callback again at a later time.
			gError("WSAEWOULDBLOCK Error on receive socket.", __LINE__, __FILE__);
		}
		else if(nRet == WSAECONNRESET)
		{
			// There is a failure on a point<->point unicast.
			cEndpoint* ep = cEndpointFactory::AllocEndpoint(ENDPOINT_TYPE_IP);
			if(!ep)
			{
				gError("Failure on remote socket during unicast, but can't alloc endpoint to report it!", __LINE__, __FILE__);
				return;
			}
			((cIPEndpoint *)ep)->mAddress = addr;
			_DeliverErrorCallback(ep, ASYNCH_ERROR_EP_FAIL);		
			ep->Release();
		}
		else
		{
			gError("Error on receive from socket!", __LINE__, __FILE__);
		}
		return;
	}
	else
	{
		// SUCCESS
		unsigned int msgType;
		if(param == (void*)&mBroadcastSocket)
		{
			msgType = MSG_TYPE_UNICAST;
		}
		else if(param == (void*)&mMulticastSocket[0])
		{
			msgType = MSG_TYPE_MULTICAST;
		}	
		buf->SetPayloadSize(nRet);
		buf->AddRef();	// Add a reference.
		this->Deliver(NULL, buf, msgType);
		buf->Release();	// Release so as to delete if possible.
	}
	return;
}

/*
 * cNetworkLayer::_SendDatagram()
 *
 * Purpose:	Sends a datagram
 * IN:		dest		-> The destination endpoint to send to
 *			buffer		-> The buffer to send
 * OUT:		-
 * Return:	The local ip address.
 */
bool cNetworkLayer::_SendDatagram(SOCKET* socket, cIPEndpoint* dest, const char *buffer, const int size)
{
	int nRet;

	// Send datagram
	while(1)
	{
		nRet = sendto(*socket, buffer, size, 0, (struct sockaddr *)&(dest->mAddress), sizeof(dest->mAddress));
		if(nRet == SOCKET_ERROR)
		{ 
			if(WSAGetLastError() == WSAEWOULDBLOCK)
			{
				continue;	// Keep trying until no longer blocked.
			}
			gError("Unable to send datagram.", __LINE__, __FILE__);
			return false;
		}
		else
		{
			return true;
		}
	}
}

/*
 * cNetworkLayer::_DeliverErrorCallback()
 *
 * Purpose:	Delivers asynch errors to any registrees.
 * IN:		obj		-> the object parameter
 *			type	-> the type parameter
 * OUT:		-
 * Return:	The local ip address.
 */
bool cNetworkLayer::_DeliverErrorCallback(cObject* obj, unsigned int type)
{
	cIterator*		iter;
	cErrorCallback* callback;

	iter = mErrorCallbackList->GetIterator();
	while(!iter->Done())
	{
		callback = (cErrorCallback *)iter->GetData();
		callback->ErrorCallback(obj, type);
		iter->GetNext();
	}
	return true;
}

/*
 * cNetworkLayer::_GetHostId()
 *
 * Purpose:	Gets the local ip address.
 * IN:		-
 * OUT:		-
 * Return:	The local ip address.
 */
LONG cNetworkLayer::_GetHostID () {

    char szLclHost [128];
    LPHOSTENT lpstHostent;
    SOCKADDR_IN stLclAddr;
    SOCKADDR_IN stRmtAddr;
    int nAddrSize = sizeof(SOCKADDR);
    SOCKET hSock;
    int nRet;
    
    /* Init local address (to zero) */
    stLclAddr.sin_addr.s_addr = INADDR_ANY;
    
    /* Get the local hostname */
    nRet = gethostname(szLclHost, sizeof(szLclHost)); 
    if (nRet != SOCKET_ERROR) {
      /* Resolve hostname for local address */
		lpstHostent = gethostbyname((LPSTR)szLclHost);
      if (lpstHostent)
        stLclAddr.sin_addr.s_addr = *((u_long FAR*) (lpstHostent->h_addr));
    } 
    
    /* If still not resolved, then try second strategy */
    if (stLclAddr.sin_addr.s_addr == INADDR_ANY) {
      /* Get a UDP socket */
      hSock = socket(AF_INET, SOCK_DGRAM, 0);
      if (hSock != INVALID_SOCKET)  {
        /* Connect to arbitrary port and address (NOT loopback) */
        stRmtAddr.sin_family = AF_INET;
        stRmtAddr.sin_port   = htons(IPPORT_ECHO);
        stRmtAddr.sin_addr.s_addr = inet_addr("128.127.50.1");
        nRet = connect(hSock,
                       (LPSOCKADDR)&stRmtAddr,
                       sizeof(SOCKADDR));
        if (nRet != SOCKET_ERROR) {
          /* Get local address */
          getsockname(hSock, 
                      (LPSOCKADDR)&stLclAddr, 
                      (int FAR*)&nAddrSize);
        }
        closesocket(hSock);   /* we're done with the socket */
      }
    }
    return (stLclAddr.sin_addr.s_addr);
} /* GetHostID() */

bool	cNetworkLayer::RegisterViewCallback(cHandle* handle, cView* callback)
{
	if(mLayerBelow)
	{
		return mLayerBelow->RegisterViewCallback(handle, callback);
	}
	else
	{
		return false;
	}
}
bool	cNetworkLayer::UnregisterViewCallback(cHandle handle)
{
	if(mLayerBelow)
	{
		return mLayerBelow->UnregisterViewCallback(handle);
	}
	else
	{
		return false;
	}
}